function [fig, ax] = emMapPlot(em, contour_type, yax, normalized)
arguments
    em struct
    contour_type {mustBeTextScalar, mustBeMember(contour_type, ["eff", "loss"])}  = "eff"
    yax string {mustBeMember(yax, ["power", "torque"])} = "torque"
    normalized = false
end
%emMapPlot
% Plot the e-machine efficiency map.
%
% Input arguments
% ---------------
% em : struct
%   engine data structure.
% contour_type : string, optional
%   Specify: 'eff' (default) for conversion efficiency;
%            'loss' for torque loss;
% yax : string, optional
%   Specify: 'torque'(default) to plot torque vs speed;
%            'power' to plot power vs speed.
% normalized : logical, optional
%   Specify: 'true' to normalize the axes and contour plot;
%            'false' (default) otherwise.
%
% Outputs
% ---------------
% fig : Figure
%   Figure handle of the plot.
% ax : Axes
%   Axes handle of the plot.

%%
if strcmp(contour_type, "loss")
    contour_type = "trqLossMap";
end
if strcmp(contour_type, "eff")
    contour_type = "effMap";
end
if isempty(yax)
    yax = "torque";
end

% Create contour plot data
spdBrk = linspace(0, em.maxSpd, 200);
switch yax
    case "torque"
        trqBrk = linspace(min(em.minTrq.Values), max(em.maxTrq.Values), 200);
        [spdMesh, trqMesh] = ndgrid(spdBrk, trqBrk);
        yMesh = trqMesh;
    case "power"
        pwrBrk = linspace(0, em.ratedPwr*1e3, 220);
        [spdMesh, pwrMesh] = ndgrid(spdBrk, pwrBrk);
        yMesh = pwrMesh .* 1e-3; % kW
        trqMesh = pwrMesh ./ spdMesh; % W
end

contour_data = em.(contour_type)(spdMesh, trqMesh); % unitless or Nm

% Remove contour plot data outside limit curves
trqMin = em.minTrq(spdMesh);
trqMax = em.maxTrq(spdMesh);

contour_data(trqMesh>trqMax) = nan;
contour_data(trqMesh<trqMin) = nan;

% Levels
switch contour_type
    case {'effMap'}
        levels = linspace(min(contour_data(:)), max(contour_data(:)), 20);
        levels = round(levels, -2);
    case {'trqLossMap'}
        levels = min(contour_data(:)) + (max(contour_data(:)) - min(contour_data(:))) .* linspace(0, 1, 20).^3;
        levels = round(levels, 0);
end

% Labels and overrides
switch contour_type
    case 'effMap'
        levels = [0.7:0.05:0.9 0.91:0.01:0.95 0.955:0.005:1];
        contour_legend_string = '\eta_{EM}';
    case 'trqLossMap'
        contour_legend_string = 'Torque loss, Nm';
end
colorScale = [levels(1) levels(end)];

%% Plot em map
fig = figure;
ax = axes;
hold on
grid on

spdBrkRPM = spdBrk .* 30/pi;
spdMeshRPM = spdMesh .* 30/pi;

% draw WOT
maxTrq = em.maxTrq(spdBrk);
minTrq = em.minTrq(spdBrk);
switch yax
    case "torque"
        yMax = maxTrq;
        yMin = minTrq;
    case "power"
        maxPwr = maxTrq .* spdBrk; % W
        yMax = maxPwr * 1e-3; % kW
        minPwr = minTrq .* spdBrk; % W
        yMin = minPwr * 1e-3; % kW
end
plot(ax, spdBrkRPM, yMax, 'k', 'LineWidth', 2, 'HandleVisibility', 'off')
plot(ax, spdBrkRPM, yMin, 'k', 'LineWidth', 2, 'HandleVisibility', 'off')

% null torque line
plot(ax, [min(spdBrkRPM) max(spdBrkRPM)], [0, 0], 'k', 'LineWidth', 1, 'HandleVisibility', 'off')

% contour plot
if normalized
    shwTxt = 'off';
else
    shwTxt = 'on';
end

[c, h] = contour(ax, spdMeshRPM, yMesh, contour_data, levels, 'ShowText', shwTxt, 'LabelSpacing', 1440);
if ~normalized
    clabel(c, h, 'LabelSpacing', 1440, 'color', 'k', 'FontWeight', 'bold')
end

caxis(colorScale)

%% Finalize figure
xlim([0 em.maxSpd].*30/pi)
switch yax
    case "torque"
        ylim([min(minTrq) max(maxTrq)]*1.05)
    case "power"
        ylim([0 max(yMax)]*1.05)
end

ylims = ylim;
xlabel('Speed, RPM')

switch yax
    case "torque"
        fill([spdBrkRPM spdBrkRPM(end) spdBrkRPM(1)], [maxTrq ylims(2) ylims(2)], 0.9*ones(1,3))
        fill([spdBrkRPM spdBrkRPM(end) spdBrkRPM(1)], [minTrq ylims(1) ylims(1)], 0.9*ones(1,3))
        ylabel('Torque, Nm')
    case "power"
        fill([spdBrkRPM spdBrkRPM(end) spdBrkRPM(1)], [yMax ylims(2) ylims(2)], 0.9*ones(1,3))
        fill([spdBrkRPM spdBrkRPM(end) spdBrkRPM(1)], [yMin ylims(1) ylims(1)], 0.9*ones(1,3))
        ylabel('Power, kW')
end

% Legend
legend(contour_legend_string)

if normalized
    ax.YTickLabel{1} = 'T_{min}';
    ax.YTickLabel(2:end-1) = {' '};
    ax.YTickLabel{end} = 'T_{max}';
    ax.XTickLabel{1} = '0';
    ax.XTickLabel(2:end-1) = {' '};
    ax.XTickLabel{end} = '\omega_{max}';
end

end